const express = require('express');
const http = require('http');
const { Server } = require('socket.io');
const cors = require('cors');
const twilio = require('twilio');
require('dotenv').config();
require('./db'); // initializes DB connection

const Contact = require('./models/Contact');
const Message = require('./models/Message');

// Initialize Twilio client (only if credentials are provided)
let twilioClient = null;
if (process.env.TWILIO_ACCOUNT_SID && process.env.TWILIO_AUTH_TOKEN) {
    twilioClient = twilio(
        process.env.TWILIO_ACCOUNT_SID,
        process.env.TWILIO_AUTH_TOKEN
    );
    console.log('Twilio client initialized');
} else {
    console.warn('WARNING: Twilio credentials not found. SMS sending will not work.');
    console.warn('Please set TWILIO_ACCOUNT_SID and TWILIO_AUTH_TOKEN in your .env file');
}

const app = express();
const server = http.createServer(app);
const io = new Server(server, {
    cors: { origin: "*" }
});

app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true })); // For Twilio webhook form data

// Serve frontend
app.use(express.static('public'));


// =========================
//       API ROUTES
// =========================

// Get all contacts
app.get('/api/contacts', async (req, res) => {
    try {
        const contacts = await Contact.getAll();
        res.json(contacts);
    } catch (err) {
        console.error("Error fetching contacts:", err);
        res.status(500).json({ error: "Internal Server Error" });
    }
});


// Get messages between two IDs
app.get('/api/messages/:id', async (req, res) => {
    try {
        const contactId = req.params.id;
        const myId = 1; // since no auth, all users are "admin id = 1"

        const messages = await Message.getByConversation(myId, contactId);
        res.json(messages);
    } catch (err) {
        console.error("Error fetching messages:", err);
        res.status(500).json({ error: "Internal Server Error" });
    }
});


// Send a message (from UI) - sends via Twilio
app.post('/send', async (req, res) => {
    try {
        const { contact_id, text } = req.body;

        if (!contact_id || !text) {
            return res.status(400).json({ error: 'contact_id and text are required' });
        }

        // Get contact to retrieve phone number
        const contact = await Contact.getById(contact_id);
        if (!contact) {
            return res.status(404).json({ error: 'Contact not found' });
        }

        if (!contact.phone) {
            return res.status(400).json({ error: 'Contact does not have a phone number' });
        }

        const myId = 1;
        const twilioPhoneNumber = process.env.TWILIO_PHONE_NUMBER;

        // Check if Twilio is configured
        if (!twilioClient) {
            const msg = await Message.create({
                sender_id: myId,
                receiver_id: contact_id,
                body: text,
                sent: 0,
                failed: 1
            });
            return res.status(500).json({ 
                error: 'Twilio not configured', 
                details: 'TWILIO_ACCOUNT_SID and TWILIO_AUTH_TOKEN must be set in .env file',
                message: msg
            });
        }

        if (!twilioPhoneNumber) {
            const msg = await Message.create({
                sender_id: myId,
                receiver_id: contact_id,
                body: text,
                sent: 0,
                failed: 1
            });
            return res.status(500).json({ 
                error: 'Twilio phone number not configured',
                details: 'TWILIO_PHONE_NUMBER must be set in .env file',
                message: msg
            });
        }

        // Send message via Twilio
        let twilioMessage;
        try {
            twilioMessage = await twilioClient.messages.create({
                body: text,
                from: twilioPhoneNumber,
                to: contact.phone
            });
            console.log('Message sent via Twilio:', twilioMessage.sid);
        } catch (twilioErr) {
            console.error('Twilio send error:', twilioErr);
            // Save message as failed
            const msg = await Message.create({
                sender_id: myId,
                receiver_id: contact_id,
                body: text,
                sent: 0,
                failed: 1
            });
            return res.status(500).json({ 
                error: 'Failed to send message via Twilio', 
                details: twilioErr.message || 'Unknown Twilio error',
                message: msg
            });
        }

        // Save message to database with Twilio message SID
        const msg = await Message.create({
            message_sid: twilioMessage.sid,
            sender_id: myId,
            receiver_id: contact_id,
            body: text,
            date_sent: new Date(twilioMessage.dateSent),
            sent: 1,
            failed: 0,
            unread: 0
        });

        // Emit real-time message
        io.to(`room_${contact_id}`).emit('chat-message', msg);

        res.json({ success: true, message: msg });

    } catch (err) {
        console.error("Message sending error:", err);
        res.status(500).json({ error: "Internal Server Error", details: err.message });
    }
});


// Twilio webhook endpoint for incoming messages
app.post('/webhook/twilio', async (req, res) => {
    try {
        const { From, To, Body, MessageSid, DateSent } = req.body;

        console.log('Incoming Twilio message:', { From, To, Body, MessageSid });

        // Find or create contact by phone number
        let contact = await Contact.getByPhone(From);
        
        if (!contact) {
            // Create new contact if doesn't exist
            const contactId = await Contact.create({
                name: From, // Use phone as name if unknown
                phone: From,
                email: null,
                role: 'Unknown',
                handle: null,
                status: 1
            });
            contact = await Contact.getById(contactId);
            console.log('Created new contact:', contact);
        }

        const myId = 1; // Admin user ID

        // Save message to database
        const msg = await Message.create({
            message_sid: MessageSid,
            sender_id: contact.id,
            receiver_id: myId,
            body: Body,
            date_sent: DateSent ? new Date(DateSent) : null,
            sent: 1,
            failed: 0,
            unread: 1
        });

        // Emit real-time message to all clients in the contact's room
        io.to(`room_${contact.id}`).emit('chat-message', msg);

        // Also emit to update contacts list if needed
        io.emit('new-message', { contactId: contact.id, message: msg });

        // Respond to Twilio (required)
        res.type('text/xml');
        res.send('<?xml version="1.0" encoding="UTF-8"?><Response></Response>');

    } catch (err) {
        console.error("Twilio webhook error:", err);
        res.type('text/xml');
        res.status(500).send('<?xml version="1.0" encoding="UTF-8"?><Response></Response>');
    }
});


// Test endpoint to simulate incoming message (for testing without Twilio webhook)
app.post('/test/incoming-message', async (req, res) => {
    try {
        const { contact_id, body } = req.body;

        if (!contact_id || !body) {
            return res.status(400).json({ error: 'contact_id and body are required' });
        }

        // Get contact
        const contact = await Contact.getById(contact_id);
        if (!contact) {
            return res.status(404).json({ error: 'Contact not found' });
        }

        const myId = 1; // Admin user ID

        // Save message to database (simulating incoming from contact)
        const msg = await Message.create({
            message_sid: `TEST_${Date.now()}`, // Dummy message SID
            sender_id: contact.id,
            receiver_id: myId,
            body: body,
            date_sent: new Date(),
            sent: 1,
            failed: 0,
            unread: 1
        });

        console.log('Test message created:', msg);

        // Emit real-time message to all clients in the contact's room
        io.to(`room_${contact.id}`).emit('chat-message', msg);

        // Also emit to update contacts list if needed
        io.emit('new-message', { contactId: contact.id, message: msg });

        res.json({ 
            success: true, 
            message: 'Test message created and emitted',
            data: msg 
        });

    } catch (err) {
        console.error("Test message error:", err);
        res.status(500).json({ error: "Internal Server Error", details: err.message });
    }
});


// =========================
//      SOCKET.IO LOGIC
// =========================

io.on('connection', socket => {
    console.log("Client connected:", socket.id);

    // User joins a contact room
    socket.on("join-room", (contactId) => {
        socket.join(`room_${contactId}`);
    });

    socket.on("disconnect", () => {
        console.log("Client disconnected:", socket.id);
    });
});


// =========================
//      START SERVER
// =========================

const PORT = 3001; // or any free port
server.listen(PORT, "0.0.0.0", () => {
    console.log(`Server running on port ${PORT}`);
});
